import { useState, useRef } from "react";
import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";
import { getLoginUrl } from "@/const";
import { Loader2, Upload, Eye, FileText } from "lucide-react";
import { Textarea } from "@/components/ui/textarea";
import { Link } from "wouter";

export default function Admin() {
  const { user, loading } = useAuth();
  const [selectedRequest, setSelectedRequest] = useState<number | null>(null);
  const [reportFile, setReportFile] = useState<File | null>(null);
  const [newPaymentMethod, setNewPaymentMethod] = useState({
    nameAr: "",
    nameEn: "",
    type: "",
    details: "",
  });
  const fileInputRef = useRef<HTMLInputElement>(null);

  const utils = trpc.useUtils();
  const { data: requests, isLoading: requestsLoading } = trpc.request.getAll.useQuery();
  const { data: paymentMethods, isLoading: methodsLoading } = trpc.paymentMethod.getAll.useQuery();
  
  const updateStatusMutation = trpc.request.updateStatus.useMutation({
    onSuccess: () => {
      toast.success("تم تحديث الحالة");
      utils.request.getAll.invalidate();
    },
  });

  const uploadReportMutation = trpc.request.uploadReport.useMutation({
    onSuccess: () => {
      toast.success("تم رفع التقرير بنجاح");
      utils.request.getAll.invalidate();
      setSelectedRequest(null);
      setReportFile(null);
    },
  });

  const createPaymentMethodMutation = trpc.paymentMethod.create.useMutation({
    onSuccess: () => {
      toast.success("تم إضافة طريقة الدفع");
      utils.paymentMethod.getAll.invalidate();
      setNewPaymentMethod({ nameAr: "", nameEn: "", type: "", details: "" });
    },
  });

  const deletePaymentMethodMutation = trpc.paymentMethod.delete.useMutation({
    onSuccess: () => {
      toast.success("تم حذف طريقة الدفع");
      utils.paymentMethod.getAll.invalidate();
    },
  });

  const handleStatusChange = async (requestId: number, status: string) => {
    await updateStatusMutation.mutateAsync({
      requestId,
      status: status as any,
    });
  };

  const handleReportUpload = async (requestId: number) => {
    if (!reportFile) {
      toast.error("الرجاء اختيار ملف");
      return;
    }

    try {
      const reader = new FileReader();
      reader.onloadend = async () => {
        const base64 = reader.result as string;
        const response = await fetch("/api/upload-report", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            file: base64,
            requestId,
          }),
        });

        if (response.ok) {
          const { url } = await response.json();
          await uploadReportMutation.mutateAsync({
            requestId,
            reportPdfUrl: url,
          });
        }
      };
      reader.readAsDataURL(reportFile);
    } catch (error) {
      toast.error("حدث خطأ أثناء رفع التقرير");
      console.error(error);
    }
  };

  const handleCreatePaymentMethod = async () => {
    if (!newPaymentMethod.nameAr || !newPaymentMethod.type || !newPaymentMethod.details) {
      toast.error("الرجاء ملء جميع الحقول المطلوبة");
      return;
    }

    await createPaymentMethodMutation.mutateAsync(newPaymentMethod);
  };

  const getStatusText = (status: string) => {
    switch (status) {
      case "pending_payment": return "في انتظار الدفع";
      case "payment_submitted": return "تم إرسال إثبات الدفع";
      case "paid": return "تم الدفع";
      case "completed": return "مكتمل";
      case "refunded": return "مسترد";
      default: return status;
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Loader2 className="h-12 w-12 animate-spin text-cyan-600" />
      </div>
    );
  }

  if (!user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-purple-50 via-white to-cyan-50">
        <Card className="max-w-md w-full">
          <CardHeader>
            <CardTitle>تسجيل الدخول مطلوب</CardTitle>
            <CardDescription>يجب تسجيل الدخول للوصول إلى لوحة التحكم</CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => window.location.href = getLoginUrl()} className="w-full">
              تسجيل الدخول
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (user.role !== "admin") {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-purple-50 via-white to-cyan-50">
        <Card className="max-w-md w-full">
          <CardHeader>
            <CardTitle>غير مصرح</CardTitle>
            <CardDescription>ليس لديك صلاحية الوصول إلى هذه الصفحة</CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50 p-8">
      <div className="max-w-7xl mx-auto">
        <div className="mb-8 flex justify-between items-center">
          <div>
            <h1 className="text-4xl font-bold text-purple-900 mb-2">لوحة التحكم</h1>
            <p className="text-gray-600">مرحباً {user.name}</p>
          </div>
          <div className="flex gap-4">
            <Link href="/dashboard">
              <Button variant="outline">الإحصائيات</Button>
            </Link>
            <Link href="/">
              <Button variant="outline">الرئيسية</Button>
            </Link>
          </div>
        </div>

        <Tabs defaultValue="requests" className="space-y-6">
          <TabsList>
            <TabsTrigger value="requests">الطلبات</TabsTrigger>
            <TabsTrigger value="payment-methods">طرق الدفع</TabsTrigger>
            <TabsTrigger value="messages">الرسائل</TabsTrigger>
          </TabsList>

          {/* Requests Tab */}
          <TabsContent value="requests">
            <Card>
              <CardHeader>
                <CardTitle>جميع الطلبات</CardTitle>
                <CardDescription>إدارة طلبات التقارير الائتمانية</CardDescription>
              </CardHeader>
              <CardContent>
                {requestsLoading ? (
                  <div className="text-center py-8">
                    <Loader2 className="h-8 w-8 animate-spin mx-auto text-cyan-600" />
                  </div>
                ) : (
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>رقم الطلب</TableHead>
                          <TableHead>البريد الإلكتروني</TableHead>
                          <TableHead>الرقم القومي</TableHead>
                          <TableHead>الحالة</TableHead>
                          <TableHead>التاريخ</TableHead>
                          <TableHead>الإجراءات</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {requests?.map((request) => (
                          <TableRow key={request.id}>
                            <TableCell>{request.id}</TableCell>
                            <TableCell>{request.email}</TableCell>
                            <TableCell>{request.idNumber}</TableCell>
                            <TableCell>
                              <Select
                                value={request.status}
                                onValueChange={(value) => handleStatusChange(request.id, value)}
                              >
                                <SelectTrigger className="w-48">
                                  <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                  <SelectItem value="pending_payment">في انتظار الدفع</SelectItem>
                                  <SelectItem value="payment_submitted">تم إرسال إثبات الدفع</SelectItem>
                                  <SelectItem value="paid">تم الدفع</SelectItem>
                                  <SelectItem value="completed">مكتمل</SelectItem>
                                  <SelectItem value="refunded">مسترد</SelectItem>
                                </SelectContent>
                              </Select>
                            </TableCell>
                            <TableCell>
                              {new Date(request.createdAt).toLocaleDateString("ar-EG")}
                            </TableCell>
                            <TableCell>
                              <div className="flex gap-2">
                                <Dialog>
                                  <DialogTrigger asChild>
                                    <Button variant="outline" size="sm">
                                      <Eye className="h-4 w-4" />
                                    </Button>
                                  </DialogTrigger>
                                  <DialogContent className="max-w-2xl">
                                    <DialogHeader>
                                      <DialogTitle>تفاصيل الطلب #{request.id}</DialogTitle>
                                    </DialogHeader>
                                    <div className="space-y-4">
                                      <div className="grid grid-cols-2 gap-4">
                                        <div>
                                          <Label>البريد الإلكتروني</Label>
                                          <p className="text-sm">{request.email}</p>
                                        </div>
                                        <div>
                                          <Label>الرقم القومي</Label>
                                          <p className="text-sm">{request.idNumber}</p>
                                        </div>
                                        <div>
                                          <Label>اسم الأم</Label>
                                          <p className="text-sm">{request.motherName}</p>
                                        </div>
                                        <div>
                                          <Label>اللغة</Label>
                                          <p className="text-sm">{request.language === "ar" ? "العربية" : "الإنجليزية"}</p>
                                        </div>
                                      </div>
                                      {request.idCardImageUrl && (
                                        <div>
                                          <Label>صورة البطاقة</Label>
                                          <img src={request.idCardImageUrl} alt="ID Card" className="mt-2 max-h-64 rounded border" />
                                        </div>
                                      )}
                                      {request.paymentProofUrl && (
                                        <div>
                                          <Label>إثبات الدفع</Label>
                                          <img src={request.paymentProofUrl} alt="Payment Proof" className="mt-2 max-h-64 rounded border" />
                                        </div>
                                      )}
                                    </div>
                                  </DialogContent>
                                </Dialog>

                                <Dialog>
                                  <DialogTrigger asChild>
                                    <Button variant="outline" size="sm" onClick={() => setSelectedRequest(request.id)}>
                                      <FileText className="h-4 w-4" />
                                    </Button>
                                  </DialogTrigger>
                                  <DialogContent>
                                    <DialogHeader>
                                      <DialogTitle>رفع التقرير</DialogTitle>
                                      <DialogDescription>رفع ملف PDF للتقرير الائتماني</DialogDescription>
                                    </DialogHeader>
                                    <div className="space-y-4">
                                      <div>
                                        <input
                                          type="file"
                                          ref={fileInputRef}
                                          onChange={(e) => setReportFile(e.target.files?.[0] || null)}
                                          accept=".pdf"
                                          className="hidden"
                                        />
                                        <Button
                                          type="button"
                                          variant="outline"
                                          onClick={() => fileInputRef.current?.click()}
                                          className="w-full"
                                        >
                                          <Upload className="ml-2 h-4 w-4" />
                                          {reportFile ? reportFile.name : "اختر ملف PDF"}
                                        </Button>
                                      </div>
                                      <Button
                                        onClick={() => handleReportUpload(request.id)}
                                        className="w-full"
                                        disabled={!reportFile || uploadReportMutation.isPending}
                                      >
                                        {uploadReportMutation.isPending ? (
                                          <>
                                            <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                                            جاري الرفع...
                                          </>
                                        ) : (
                                          "رفع التقرير"
                                        )}
                                      </Button>
                                    </div>
                                  </DialogContent>
                                </Dialog>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Payment Methods Tab */}
          <TabsContent value="payment-methods">
            <div className="grid gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>إضافة طريقة دفع جديدة</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label>الاسم بالعربية</Label>
                      <Input
                        value={newPaymentMethod.nameAr}
                        onChange={(e) => setNewPaymentMethod({ ...newPaymentMethod, nameAr: e.target.value })}
                        placeholder="فودافون كاش"
                      />
                    </div>
                    <div>
                      <Label>الاسم بالإنجليزية</Label>
                      <Input
                        value={newPaymentMethod.nameEn}
                        onChange={(e) => setNewPaymentMethod({ ...newPaymentMethod, nameEn: e.target.value })}
                        placeholder="Vodafone Cash"
                      />
                    </div>
                  </div>
                  <div>
                    <Label>النوع</Label>
                    <Input
                      value={newPaymentMethod.type}
                      onChange={(e) => setNewPaymentMethod({ ...newPaymentMethod, type: e.target.value })}
                      placeholder="vodafone_cash"
                    />
                  </div>
                  <div>
                    <Label>التفاصيل</Label>
                    <Textarea
                      value={newPaymentMethod.details}
                      onChange={(e) => setNewPaymentMethod({ ...newPaymentMethod, details: e.target.value })}
                      placeholder="رقم الموبايل: 01234567890"
                      rows={3}
                    />
                  </div>
                  <Button onClick={handleCreatePaymentMethod} disabled={createPaymentMethodMutation.isPending}>
                    {createPaymentMethodMutation.isPending ? (
                      <>
                        <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                        جاري الإضافة...
                      </>
                    ) : (
                      "إضافة طريقة الدفع"
                    )}
                  </Button>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>طرق الدفع الحالية</CardTitle>
                </CardHeader>
                <CardContent>
                  {methodsLoading ? (
                    <div className="text-center py-8">
                      <Loader2 className="h-8 w-8 animate-spin mx-auto text-cyan-600" />
                    </div>
                  ) : (
                    <div className="space-y-4">
                      {paymentMethods?.map((method) => (
                        <div key={method.id} className="p-4 border rounded-lg flex justify-between items-start">
                          <div>
                            <h3 className="font-bold">{method.nameAr}</h3>
                            <p className="text-sm text-gray-600">{method.type}</p>
                            <p className="text-sm mt-2 whitespace-pre-line">{method.details}</p>
                            <p className="text-sm mt-2">
                              <span className={method.isActive ? "text-green-600" : "text-red-600"}>
                                {method.isActive ? "نشط" : "غير نشط"}
                              </span>
                            </p>
                          </div>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => deletePaymentMethodMutation.mutate({ id: method.id })}
                          >
                            حذف
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Messages Tab */}
          <TabsContent value="messages">
            <Card>
              <CardHeader>
                <CardTitle>الرسائل</CardTitle>
                <CardDescription>رسائل الدردشة من المستخدمين</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {requests?.map((request) => (
                    <Dialog key={request.id}>
                      <DialogTrigger asChild>
                        <div className="p-4 border rounded-lg cursor-pointer hover:bg-gray-50">
                          <div className="flex justify-between items-center">
                            <div>
                              <h3 className="font-bold">طلب #{request.id}</h3>
                              <p className="text-sm text-gray-600">{request.email}</p>
                            </div>
                            <Button variant="outline" size="sm">عرض الرسائل</Button>
                          </div>
                        </div>
                      </DialogTrigger>
                      <DialogContent className="max-w-2xl">
                        <DialogHeader>
                          <DialogTitle>رسائل الطلب #{request.id}</DialogTitle>
                        </DialogHeader>
                        <MessagesView requestId={request.id} />
                      </DialogContent>
                    </Dialog>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}

function MessagesView({ requestId }: { requestId: number }) {
  const [message, setMessage] = useState("");
  const { data: messages, isLoading, refetch } = trpc.message.getByRequest.useQuery({ requestId });
  const sendMessageMutation = trpc.message.sendAdmin.useMutation({
    onSuccess: () => {
      setMessage("");
      refetch();
    },
  });

  const handleSend = async () => {
    if (!message.trim()) {
      toast.error("الرجاء كتابة رسالة");
      return;
    }

    await sendMessageMutation.mutateAsync({
      requestId,
      message: message.trim(),
    });
  };

  return (
    <div className="space-y-4">
      <div className="max-h-96 overflow-y-auto space-y-3">
        {isLoading ? (
          <div className="text-center py-8">
            <Loader2 className="h-8 w-8 animate-spin mx-auto text-cyan-600" />
          </div>
        ) : messages && messages.length > 0 ? (
          messages.map((msg) => (
            <div
              key={msg.id}
              className={`flex ${msg.senderType === "admin" ? "justify-end" : "justify-start"}`}
            >
              <div
                className={`max-w-[80%] rounded-lg p-3 ${
                  msg.senderType === "admin"
                    ? "bg-purple-600 text-white"
                    : "bg-gray-200 text-gray-900"
                }`}
              >
                <p className="text-sm">{msg.message}</p>
                <p className="text-xs opacity-70 mt-1">
                  {new Date(msg.createdAt).toLocaleString("ar-EG")}
                </p>
              </div>
            </div>
          ))
        ) : (
          <p className="text-center text-gray-500">لا توجد رسائل</p>
        )}
      </div>
      <div className="flex gap-2 border-t pt-4">
        <Input
          value={message}
          onChange={(e) => setMessage(e.target.value)}
          placeholder="اكتب رسالتك..."
          onKeyPress={(e) => {
            if (e.key === "Enter") {
              handleSend();
            }
          }}
        />
        <Button onClick={handleSend} disabled={sendMessageMutation.isPending}>
          {sendMessageMutation.isPending ? (
            <Loader2 className="h-4 w-4 animate-spin" />
          ) : (
            "إرسال"
          )}
        </Button>
      </div>
    </div>
  );
}
