import { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";
import { Link } from "wouter";
import { APP_LOGO } from "@/const";
import { BarChart3, TrendingUp, Users, FileText, Edit2, Save, X } from "lucide-react";

export default function Dashboard() {
  const { data: stats, refetch } = trpc.statistics.get.useQuery();
  const updateStatsMutation = trpc.statistics.update.useMutation();
  const [editing, setEditing] = useState(false);
  const [formData, setFormData] = useState({
    totalRequests: 0,
    totalRevenue: 0,
    totalUsers: 0,
  });

  useEffect(() => {
    if (stats) {
      setFormData({
        totalRequests: stats.totalRequests,
        totalRevenue: stats.totalRevenue,
        totalUsers: stats.totalUsers,
      });
    }
  }, [stats]);

  const handleSave = async () => {
    try {
      await updateStatsMutation.mutateAsync(formData);
      toast.success("تم تحديث الإحصائيات بنجاح");
      setEditing(false);
      refetch();
    } catch (error) {
      toast.error("حدث خطأ أثناء تحديث الإحصائيات");
    }
  };

  const chartData = [
    { label: "الطلبات", value: stats?.totalRequests || 0, color: "bg-cyan-500" },
    { label: "الإيرادات (جنيه)", value: stats?.totalRevenue || 0, color: "bg-purple-500" },
    { label: "المستخدمين", value: stats?.totalUsers || 0, color: "bg-orange-500" },
  ];

  const maxValue = Math.max(...chartData.map(d => d.value), 1);

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link href="/">
            <div className="flex items-center gap-3 cursor-pointer">
              <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
            </div>
          </Link>
          <div className="flex gap-4">
            <Link href="/admin">
              <Button variant="outline">لوحة التحكم</Button>
            </Link>
            <Link href="/">
              <Button variant="outline">الرئيسية</Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-6xl mx-auto">
          <div className="flex justify-between items-center mb-8">
            <h1 className="text-4xl font-bold text-purple-900">لوحة الإحصائيات</h1>
            {!editing ? (
              <Button onClick={() => setEditing(true)}>
                <Edit2 className="ml-2 h-4 w-4" />
                تعديل الإحصائيات
              </Button>
            ) : (
              <div className="flex gap-2">
                <Button onClick={handleSave} disabled={updateStatsMutation.isPending}>
                  <Save className="ml-2 h-4 w-4" />
                  حفظ
                </Button>
                <Button variant="outline" onClick={() => setEditing(false)}>
                  <X className="ml-2 h-4 w-4" />
                  إلغاء
                </Button>
              </div>
            )}
          </div>

          {/* Stats Cards */}
          <div className="grid md:grid-cols-3 gap-6 mb-12">
            <Card className="border-t-4 border-t-cyan-500">
              <CardHeader className="flex flex-row items-center justify-between pb-2">
                <CardTitle className="text-lg font-medium">إجمالي الطلبات</CardTitle>
                <FileText className="h-5 w-5 text-cyan-600" />
              </CardHeader>
              <CardContent>
                {editing ? (
                  <Input
                    type="number"
                    value={formData.totalRequests}
                    onChange={(e) => setFormData({ ...formData, totalRequests: parseInt(e.target.value) || 0 })}
                    className="text-3xl font-bold"
                  />
                ) : (
                  <div className="text-3xl font-bold text-cyan-600">{stats?.totalRequests || 0}</div>
                )}
                <p className="text-sm text-gray-600 mt-2">عدد الطلبات الكلي</p>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-purple-500">
              <CardHeader className="flex flex-row items-center justify-between pb-2">
                <CardTitle className="text-lg font-medium">إجمالي الإيرادات</CardTitle>
                <TrendingUp className="h-5 w-5 text-purple-600" />
              </CardHeader>
              <CardContent>
                {editing ? (
                  <Input
                    type="number"
                    value={formData.totalRevenue}
                    onChange={(e) => setFormData({ ...formData, totalRevenue: parseInt(e.target.value) || 0 })}
                    className="text-3xl font-bold"
                  />
                ) : (
                  <div className="text-3xl font-bold text-purple-600">{stats?.totalRevenue || 0} جنيه</div>
                )}
                <p className="text-sm text-gray-600 mt-2">الإيرادات الكلية</p>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-orange-500">
              <CardHeader className="flex flex-row items-center justify-between pb-2">
                <CardTitle className="text-lg font-medium">عدد المستخدمين</CardTitle>
                <Users className="h-5 w-5 text-orange-600" />
              </CardHeader>
              <CardContent>
                {editing ? (
                  <Input
                    type="number"
                    value={formData.totalUsers}
                    onChange={(e) => setFormData({ ...formData, totalUsers: parseInt(e.target.value) || 0 })}
                    className="text-3xl font-bold"
                  />
                ) : (
                  <div className="text-3xl font-bold text-orange-600">{stats?.totalUsers || 0}</div>
                )}
                <p className="text-sm text-gray-600 mt-2">المستخدمين المسجلين</p>
              </CardContent>
            </Card>
          </div>

          {/* Bar Chart */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BarChart3 className="h-6 w-6" />
                الرسم البياني للإحصائيات
              </CardTitle>
              <CardDescription>عرض مرئي للبيانات الإحصائية</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {chartData.map((item, index) => (
                  <div key={index} className="space-y-2">
                    <div className="flex justify-between items-center">
                      <span className="font-medium text-gray-700">{item.label}</span>
                      <span className="font-bold text-lg">{item.value.toLocaleString()}</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-8 overflow-hidden">
                      <div
                        className={`${item.color} h-full rounded-full transition-all duration-500 flex items-center justify-end px-4`}
                        style={{ width: `${(item.value / maxValue) * 100}%` }}
                      >
                        {item.value > 0 && (
                          <span className="text-white text-sm font-semibold">
                            {((item.value / maxValue) * 100).toFixed(0)}%
                          </span>
                        )}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
