import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { APP_LOGO } from "@/const";
import { Link } from "wouter";
import { CheckCircle2, FileText, Shield, TrendingUp, Star, MessageCircle, ChevronDown, ChevronUp, Loader2, Download } from "lucide-react";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";

export default function Home() {
  const [showDetails, setShowDetails] = useState(false);
  const [showReviewForm, setShowReviewForm] = useState(false);
  const [rating, setRating] = useState(0);
  const [reviewForm, setReviewForm] = useState({ name: "", comment: "" });
  const [expandedFaq, setExpandedFaq] = useState<number | null>(null);
  const [showAllReviews, setShowAllReviews] = useState(false);
  
  const { data: reviews } = trpc.review.getAll.useQuery();
  const createReviewMutation = trpc.review.create.useMutation({
    onSuccess: () => {
      toast.success("شكراً لتقييمك!");
      setShowReviewForm(false);
      setRating(0);
      setReviewForm({ name: "", comment: "" });
    },
  });
  
  const handleSubmitReview = async () => {
    if (!rating || !reviewForm.name) {
      toast.error("الرجاء إدخال الاسم والتقييم");
      return;
    }
    await createReviewMutation.mutateAsync({
      name: reviewForm.name,
      rating,
      comment: reviewForm.comment,
    });
  };
  
  const faqItems = [
    {
      question: "ما هو التقرير الائتماني؟",
      answer: "التقرير الائتماني هو ملف يحتوي على جميع معلوماتك المالية والائتمانية، بما في ذلك القروض والبطاقات الائتمانية والتسهيلات البنكية وتاريخ السداد."
    },
    {
      question: "لماذا أحتاج للتقرير الائتماني؟",
      answer: "يساعدك التقرير الائتماني في معرفة موقفك الائتماني الكامل، تحسين فرص الحصول على التمويل، تجنب المفاجآت البنكية، واتخاذ قرارات مالية ذكية."
    },
    {
      question: "كم يستغرق الحصول على التقرير؟",
      answer: "بعد إتمام عملية الدفع، يتم مراجعة طلبك وإرسال التقرير في غضون 24-48 ساعة عبر البريد الإلكتروني."
    },
    {
      question: "ما هي طرق الدفع المتاحة؟",
      answer: "نقبل الدفع عبر فودافون كاش، بطاقات الائتمان والخصم، والتحويل البنكي المباشر."
    },
    {
      question: "هل بياناتي آمنة؟",
      answer: "نعم، جميع بياناتك محمية ومشفرة وفقاً لأعلى معايير الأمن والخصوصية. لن يتم مشاركة بياناتك مع أي طرف ثالث."
    },
    {
      question: "ماذا أفعل إذا وجدت خطأ في التقرير؟",
      answer: "يمكنك التواصل معنا مباشرة عبر نظام الدردشة أو البريد الإلكتروني لمراجعة التقرير وتصحيح أي أخطاء."
    },
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <div className="flex items-center gap-3">
            <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
          </div>
          <div className="flex gap-2">
            <a href="/iscore-guide.pdf" download>
              <Button variant="outline">
                <Download className="ml-2 h-4 w-4" />
                دليل الخدمة
              </Button>
            </a>
            <Link href="/track">
              <Button variant="outline">تتبع الطلب</Button>
            </Link>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="py-16 md:py-24">
        <div className="container mx-auto px-4">
          <div className="text-center max-w-4xl mx-auto">
            <h1 className="text-4xl md:text-6xl font-bold text-purple-900 mb-6">
              استعلم عن تقريرك الائتماني
            </h1>
            <p className="text-xl md:text-2xl text-gray-700 mb-8">
              احصل على تقريرك الائتماني الشامل بسعر مخفض
            </p>
            
            {/* Price Badge */}
            <div className="inline-block bg-gradient-to-r from-red-500 to-red-600 text-white rounded-full px-8 py-4 mb-8 shadow-lg">
              <div className="text-5xl font-bold mb-1">180 جنيه</div>
              <div className="text-lg line-through opacity-90">بدلاً من 450 جنيه</div>
              <div className="text-2xl font-semibold mt-1">خصم 60%</div>
            </div>

            {/* CTA Buttons */}
            <div className="flex flex-col sm:flex-row gap-4 justify-center items-center mb-12">
              <Link href="/request">
                <Button size="lg" className="text-xl px-8 py-6 bg-cyan-500 hover:bg-cyan-600 text-white">
                  استعلم الآن
                </Button>
              </Link>
              <Dialog open={showDetails} onOpenChange={setShowDetails}>
                <DialogTrigger asChild>
                  <Button size="lg" variant="outline" className="text-xl px-8 py-6">
                    المزيد عن الخدمة
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-3xl max-h-[80vh] overflow-y-auto">
                  <DialogHeader>
                    <DialogTitle className="text-2xl">معلومات عن خدمة التقرير الائتماني</DialogTitle>
                    <DialogDescription className="text-base">
                      كل ما تحتاج معرفته عن التقرير الائتماني من iScore
                    </DialogDescription>
                  </DialogHeader>
                  <div className="space-y-4 text-right">
                    <div>
                      <h3 className="font-bold text-lg mb-2">ما هو التقرير الائتماني؟</h3>
                      <p className="text-gray-700">
                        التقرير الائتماني هو وثيقة تحتوي على معلومات شاملة عن تاريخك المالي والائتماني، 
                        ويساعدك على معرفة موقفك الائتماني الكامل وفرص الحصول على التمويل.
                      </p>
                    </div>
                    <div>
                      <h3 className="font-bold text-lg mb-2">ماذا يتضمن التقرير؟</h3>
                      <ul className="list-disc list-inside space-y-2 text-gray-700">
                        <li>معرفة موقفك الائتماني الكامل</li>
                        <li>تحسين فرص الحصول على التمويل</li>
                        <li>تجنب المفاجآت البنكية</li>
                        <li>اتخاذ قرارات مالية ذكية</li>
                      </ul>
                    </div>
                    <div>
                      <h3 className="font-bold text-lg mb-2">شروط وأحكام الخدمة</h3>
                      <ul className="list-disc list-inside space-y-2 text-gray-700 text-sm">
                        <li>يتحمل المستخدم كامل المسؤولية عن صحة البيانات المقدمة</li>
                        <li>يتم تسليم التقرير خلال 24-48 ساعة من استلام الدفع</li>
                        <li>يتم خصم الرسوم بالكامل فور تقديم الطلب</li>
                        <li>لا يجوز استرداد الرسوم بعد تسليم التقرير</li>
                        <li>يحق للشركة رفض الطلب في حالة عدم استيفاء الشروط</li>
                      </ul>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4 w-16 h-16 bg-cyan-100 rounded-full flex items-center justify-center">
                  <FileText className="w-8 h-8 text-cyan-600" />
                </div>
                <CardTitle>معرفة موقفك الائتماني الكامل</CardTitle>
              </CardHeader>
            </Card>
            
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4 w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center">
                  <TrendingUp className="w-8 h-8 text-purple-600" />
                </div>
                <CardTitle>تحسين فرص الحصول على التمويل</CardTitle>
              </CardHeader>
            </Card>
            
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4 w-16 h-16 bg-green-100 rounded-full flex items-center justify-center">
                  <Shield className="w-8 h-8 text-green-600" />
                </div>
                <CardTitle>تجنب المفاجآت البنكية</CardTitle>
              </CardHeader>
            </Card>
            
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4 w-16 h-16 bg-orange-100 rounded-full flex items-center justify-center">
                  <CheckCircle2 className="w-8 h-8 text-orange-600" />
                </div>
                <CardTitle>اتخاذ قرارات مالية ذكية</CardTitle>
              </CardHeader>
            </Card>
          </div>
        </div>
      </section>

      {/* Payment Methods */}
      <section className="py-16">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl font-bold text-purple-900 mb-8">طرق الدفع المتاحة</h2>
          <p className="text-gray-600 mb-6">نقبل الدفع عبر فودافون كاش، بطاقات الائتمان، والتحويل البنكي</p>
          <div className="flex justify-center gap-4 flex-wrap">
            <img src="https://www.vodafone.com.eg/vodafoneportalWeb/images/logo.svg" alt="Vodafone" className="h-12" />
            <img src="https://upload.wikimedia.org/wikipedia/commons/5/5e/Visa_Inc._logo.svg" alt="Visa" className="h-12" />
            <img src="https://upload.wikimedia.org/wikipedia/commons/2/2a/Mastercard-logo.svg" alt="Mastercard" className="h-12" />
          </div>
        </div>
      </section>

      {/* Reviews Section */}
      <section className="bg-white py-16">
        <div className="container">
          <h2 className="text-3xl font-bold text-center text-purple-900 mb-12">آراء عملائنا</h2>
          
          <div className="grid md:grid-cols-3 gap-6 mb-8">
            {(showAllReviews ? reviews : reviews?.slice(0, 3))?.map((review) => (
              <Card key={review.id} className="border-t-4 border-t-cyan-500">
                <CardContent className="pt-6">
                  <div className="flex gap-1 mb-3">
                    {[...Array(5)].map((_, i) => (
                      <Star
                        key={i}
                        className={`h-5 w-5 ${i < review.rating ? 'fill-yellow-400 text-yellow-400' : 'text-gray-300'}`}
                      />
                    ))}
                  </div>
                  <p className="text-gray-700 mb-4">{review.comment}</p>
                  <p className="font-semibold text-purple-900">{review.name}</p>
                </CardContent>
              </Card>
            ))}
          </div>

          {reviews && reviews.length > 3 && (
            <div className="text-center mb-8">
              <Button
                onClick={() => setShowAllReviews(!showAllReviews)}
                variant="outline"
                size="lg"
                className="gap-2"
              >
                {showAllReviews ? (
                  <>
                    <ChevronUp className="h-4 w-4" />
                    إخفاء
                  </>
                ) : (
                  <>
                    <ChevronDown className="h-4 w-4" />
                    عرض المزيد ({reviews.length - 3} تقييم)
                  </>
                )}
              </Button>
            </div>
          )}

          <div className="text-center">
            {!showReviewForm ? (
              <Button onClick={() => setShowReviewForm(true)} size="lg">
                <MessageCircle className="ml-2 h-5 w-5" />
                قيّم خدمتنا
              </Button>
            ) : (
              <Card className="max-w-md mx-auto">
                <CardHeader>
                  <CardTitle>قيّم خدمتنا</CardTitle>
                  <CardDescription>شاركنا تجربتك</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label>الاسم</Label>
                    <Input
                      value={reviewForm.name}
                      onChange={(e) => setReviewForm({ ...reviewForm, name: e.target.value })}
                      placeholder="أدخل اسمك"
                    />
                  </div>
                  <div>
                    <Label>التقييم</Label>
                    <div className="flex gap-2 mt-2">
                      {[1, 2, 3, 4, 5].map((star) => (
                        <Star
                          key={star}
                          className={`h-8 w-8 cursor-pointer transition-colors ${
                            star <= rating ? 'fill-yellow-400 text-yellow-400' : 'text-gray-300 hover:text-yellow-200'
                          }`}
                          onClick={() => setRating(star)}
                        />
                      ))}
                    </div>
                  </div>
                  <div>
                    <Label>التعليق (اختياري)</Label>
                    <Input
                      value={reviewForm.comment}
                      onChange={(e) => setReviewForm({ ...reviewForm, comment: e.target.value })}
                      placeholder="شاركنا رأيك"
                    />
                  </div>
                  <div className="flex gap-2">
                    <Button onClick={handleSubmitReview} className="flex-1" disabled={createReviewMutation.isPending}>
                      {createReviewMutation.isPending ? (
                        <>
                          <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                          جاري الإرسال...
                        </>
                      ) : (
                        "إرسال"
                      )}
                    </Button>
                    <Button variant="outline" onClick={() => setShowReviewForm(false)}>
                      إلغاء
                    </Button>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="bg-gradient-to-br from-purple-50 to-cyan-50 py-16">
        <div className="container">
          <h2 className="text-3xl font-bold text-center text-purple-900 mb-12">الأسئلة الشائعة</h2>
          <div className="max-w-3xl mx-auto space-y-4">
            {faqItems.map((item, index) => (
              <Card key={index} className="overflow-hidden">
                <button
                  className="w-full p-6 flex justify-between items-center text-right hover:bg-gray-50 transition-colors"
                  onClick={() => setExpandedFaq(expandedFaq === index ? null : index)}
                >
                  <h3 className="text-lg font-semibold text-purple-900 flex-1">{item.question}</h3>
                  {expandedFaq === index ? (
                    <ChevronUp className="h-5 w-5 text-cyan-600 flex-shrink-0 mr-4" />
                  ) : (
                    <ChevronDown className="h-5 w-5 text-gray-400 flex-shrink-0 mr-4" />
                  )}
                </button>
                {expandedFaq === index && (
                  <div className="px-6 pb-6 text-gray-700 leading-relaxed">
                    {item.answer}
                  </div>
                )}
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-purple-900 text-white py-8">
        <div className="container mx-auto px-4 text-center">
          <p className="text-lg">© 2025 iScore - جميع الحقوق محفوظة</p>
          <p className="text-sm mt-2 opacity-90">خدمة التقرير الائتماني الموثوقة</p>
        </div>
      </footer>
    </div>
  );
}
