import { useState, useRef } from "react";
import { useParams, useLocation, Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";
import { APP_LOGO } from "@/const";
import { Copy, Upload, Loader2, CheckCircle2 } from "lucide-react";

export default function Payment() {
  const { id } = useParams();
  const [, setLocation] = useLocation();
  const [selectedMethod, setSelectedMethod] = useState<number | null>(null);
  const [paymentDetails, setPaymentDetails] = useState("");
  const [paymentProofFile, setPaymentProofFile] = useState<File | null>(null);
  const [paymentProofPreview, setPaymentProofPreview] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const { data: paymentMethods, isLoading } = trpc.paymentMethod.getActive.useQuery();
  const updatePaymentMutation = trpc.request.updatePayment.useMutation();

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setPaymentProofFile(file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setPaymentProofPreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    toast.success("تم النسخ");
  };

  const handleSubmit = async () => {
    if (!selectedMethod) {
      toast.error("الرجاء اختيار طريقة الدفع");
      return;
    }
    if (!paymentDetails) {
      toast.error("الرجاء إدخال تفاصيل الدفع");
      return;
    }
    if (!paymentProofFile) {
      toast.error("الرجاء رفع إثبات الدفع");
      return;
    }

    setIsSubmitting(true);

    try {
      // Upload payment proof
      const reader = new FileReader();
      reader.onloadend = async () => {
        const base64 = reader.result as string;
        const response = await fetch("/api/upload-payment-proof", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            file: base64,
            requestId: id,
          }),
        });

        if (response.ok) {
          const { url } = await response.json();
          
          const selectedMethodData = paymentMethods?.find(m => m.id === selectedMethod);
          
          await updatePaymentMutation.mutateAsync({
            requestId: parseInt(id || "0"),
            paymentMethod: selectedMethodData?.nameAr || "",
            paymentDetails,
            paymentProofUrl: url,
          });

          toast.success("تم إرسال الطلب بنجاح");
          setLocation(`/payment-result/${id}`);
        }
      };
      reader.readAsDataURL(paymentProofFile);
    } catch (error) {
      toast.error("حدث خطأ أثناء إرسال الطلب");
      console.error(error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const selectedMethodData = paymentMethods?.find(m => m.id === selectedMethod);

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link href="/">
            <div className="flex items-center gap-3 cursor-pointer">
              <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
            </div>
          </Link>
          <Link href="/track">
            <Button variant="outline">تتبع الطلب</Button>
          </Link>
        </div>
      </header>

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-3xl mx-auto">
          <Card>
            <CardHeader>
              <CardTitle className="text-2xl">خيارات الدفع</CardTitle>
              <CardDescription>اختر طريقة الدفع المفضلة</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              {/* Cost Display */}
              <div className="bg-cyan-50 p-6 rounded-lg text-center">
                <p className="text-lg mb-2">تكلفة الخدمة</p>
                <p className="text-4xl font-bold text-cyan-600">20 جنيه</p>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => copyToClipboard("20")}
                  className="mt-2"
                >
                  <Copy className="ml-2 h-4 w-4" />
                  نسخ المبلغ
                </Button>
              </div>

              {/* Payment Methods */}
              {isLoading ? (
                <div className="text-center py-8">
                  <Loader2 className="h-8 w-8 animate-spin mx-auto text-cyan-600" />
                </div>
              ) : (
                <div className="space-y-4">
                  <Label className="text-lg">اختر طريقة الدفع:</Label>
                  {paymentMethods?.map((method) => (
                    <div
                      key={method.id}
                      onClick={() => setSelectedMethod(method.id)}
                      className={`p-4 border-2 rounded-lg cursor-pointer transition-all ${
                        selectedMethod === method.id
                          ? "border-cyan-600 bg-cyan-50"
                          : "border-gray-200 hover:border-cyan-300"
                      }`}
                    >
                      <div className="flex items-center justify-between">
                        <div>
                          <h3 className="font-bold text-lg">{method.nameAr}</h3>
                          <p className="text-sm text-gray-600">{method.type}</p>
                        </div>
                        {selectedMethod === method.id && (
                          <CheckCircle2 className="h-6 w-6 text-cyan-600" />
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              )}

              {/* Payment Details */}
              {selectedMethod && selectedMethodData && (
                <div className="bg-purple-50 p-6 rounded-lg space-y-4">
                  <h3 className="font-bold text-lg">تفاصيل الدفع</h3>
                  <div className="bg-white p-4 rounded border">
                    <p className="text-gray-700 whitespace-pre-line">{selectedMethodData.details}</p>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => copyToClipboard(selectedMethodData.details)}
                      className="mt-2"
                    >
                      <Copy className="ml-2 h-4 w-4" />
                      نسخ التفاصيل
                    </Button>
                  </div>
                  <p className="text-sm text-gray-600">
                    قم بدفع المبلغ إلى أحد تفاصيل الدفع أعلاه
                  </p>
                </div>
              )}

              {/* Payment Proof */}
              {selectedMethod && (
                <>
                  <div>
                    <Label htmlFor="paymentDetails">تفاصيل الدفع (الرقم أو الحساب الذي حولت منه)</Label>
                    <Textarea
                      id="paymentDetails"
                      value={paymentDetails}
                      onChange={(e) => setPaymentDetails(e.target.value)}
                      placeholder="أدخل تفاصيل الدفع"
                      rows={3}
                    />
                  </div>

                  <div>
                    <Label htmlFor="paymentProof">رفع إثبات الدفع</Label>
                    <div className="mt-2">
                      <input
                        type="file"
                        ref={fileInputRef}
                        onChange={handleFileChange}
                        accept="image/*"
                        className="hidden"
                        id="paymentProof"
                      />
                      <Button
                        type="button"
                        variant="outline"
                        onClick={() => fileInputRef.current?.click()}
                        className="w-full"
                      >
                        <Upload className="ml-2 h-4 w-4" />
                        {paymentProofFile ? "تم رفع الصورة" : "رفع إثبات الدفع"}
                      </Button>
                      {paymentProofPreview && (
                        <img src={paymentProofPreview} alt="Preview" className="mt-4 max-h-48 mx-auto rounded border" />
                      )}
                    </div>
                  </div>

                  <Button
                    onClick={handleSubmit}
                    className="w-full"
                    size="lg"
                    disabled={isSubmitting}
                  >
                    {isSubmitting ? (
                      <>
                        <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                        جاري الإرسال...
                      </>
                    ) : (
                      "تم الدفع"
                    )}
                  </Button>
                </>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
