import { useState, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Progress } from "@/components/ui/progress";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";
import { useLocation } from "wouter";
import { APP_LOGO } from "@/const";
import { Link } from "wouter";
import { ArrowRight, Upload, Loader2 } from "lucide-react";
import { storagePut } from "../../../server/storage";

export default function Request() {
  const [, setLocation] = useLocation();
  const [step, setStep] = useState(1);
  const [language, setLanguage] = useState("ar");
  const [formData, setFormData] = useState({
    motherName: "",
    idNumber: "",
    email: "",
    phoneNumber: "",
  });
  const [couponCode, setCouponCode] = useState("");
  const [discount, setDiscount] = useState(0);
  const [finalPrice, setFinalPrice] = useState(180);
  const [couponError, setCouponError] = useState("");
  const [idCardFile, setIdCardFile] = useState<File | null>(null);
  const [idCardPreview, setIdCardPreview] = useState<string>("");
  const [requestId, setRequestId] = useState<number | null>(null);
  const [verificationCode, setVerificationCode] = useState("");
  const fileInputRef = useRef<HTMLInputElement>(null);

  const createRequestMutation = trpc.request.create.useMutation();
  const updateIdCardMutation = trpc.request.updateIdCard.useMutation();
  const validateCouponQuery = trpc.coupon.validate.useQuery(
    { code: couponCode },
    { enabled: false }
  );

  const handleApplyCoupon = async () => {
    if (!couponCode.trim()) {
      setCouponError("الرجاء إدخال كود الخصم");
      return;
    }

    const result = await validateCouponQuery.refetch();
    if (result.data?.valid && result.data.coupon) {
      const coupon = result.data.coupon;
      let discountAmount = 0;
      
      if (coupon.discountType === "percentage") {
        discountAmount = Math.floor((180 * coupon.discountValue) / 100);
      } else {
        discountAmount = coupon.discountValue;
      }
      
      setDiscount(discountAmount);
      setFinalPrice(180 - discountAmount);
      setCouponError("");
      toast.success(`تم تطبيق الخصم: ${discountAmount} جنيه`);
    } else {
      setCouponError(result.data?.message || "كود الخصم غير صحيح");
      setDiscount(0);
      setFinalPrice(180);
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setIdCardFile(file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setIdCardPreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleStep1Submit = async () => {
    if (!language) {
      toast.error("الرجاء اختيار اللغة");
      return;
    }
    setStep(2);
  };

  const handleStep2Submit = async () => {
    if (!idCardFile) {
      toast.error("الرجاء رفع صورة البطاقة");
      return;
    }
    if (!formData.motherName || !formData.idNumber || !formData.email || !formData.phoneNumber) {
      toast.error("الرجاء ملء جميع الحقول المطلوبة");
      return;
    }

    try {
      // Create request first
      const result = await createRequestMutation.mutateAsync({
        language,
        ...formData,
      });

      if (result.success && result.requestId) {
        setRequestId(result.requestId);
        setVerificationCode(result.verificationCode || "");

        // Upload ID card image
        const reader = new FileReader();
        reader.onloadend = async () => {
          const base64 = reader.result as string;
          const response = await fetch("/api/upload-id-card", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
              file: base64,
              requestId: result.requestId,
            }),
          });

          if (response.ok) {
            const { url } = await response.json();
            await updateIdCardMutation.mutateAsync({
              requestId: result.requestId!,
              idCardImageUrl: url,
            });
            setStep(3);
          }
        };
        reader.readAsDataURL(idCardFile);
      }
    } catch (error) {
      toast.error("حدث خطأ أثناء إنشاء الطلب");
      console.error(error);
    }
  };

  const progressValue = (step / 4) * 100;

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link href="/">
            <div className="flex items-center gap-3 cursor-pointer">
              <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
            </div>
          </Link>
          <Link href="/track">
            <Button variant="outline">تتبع الطلب</Button>
          </Link>
        </div>
      </header>

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-2xl mx-auto">
          {/* Progress Steps */}
          <div className="mb-8">
            <div className="flex justify-between mb-4">
              <div className={`flex items-center ${step >= 1 ? 'text-cyan-600' : 'text-gray-400'}`}>
                <div className={`w-8 h-8 rounded-full flex items-center justify-center ${step >= 1 ? 'bg-cyan-600 text-white' : 'bg-gray-300'}`}>
                  {step > 1 ? '✓' : '1'}
                </div>
                <span className="mr-2 text-sm">بيانات الطلب</span>
              </div>
              <div className={`flex items-center ${step >= 2 ? 'text-cyan-600' : 'text-gray-400'}`}>
                <div className={`w-8 h-8 rounded-full flex items-center justify-center ${step >= 2 ? 'bg-cyan-600 text-white' : 'bg-gray-300'}`}>
                  {step > 2 ? '✓' : '2'}
                </div>
                <span className="mr-2 text-sm">الدفع</span>
              </div>
              <div className={`flex items-center ${step >= 3 ? 'text-cyan-600' : 'text-gray-400'}`}>
                <div className={`w-8 h-8 rounded-full flex items-center justify-center ${step >= 3 ? 'bg-cyan-600 text-white' : 'bg-gray-300'}`}>
                  {step > 3 ? '✓' : '3'}
                </div>
                <span className="mr-2 text-sm">نتيجة الدفع</span>
              </div>
            </div>
            <Progress value={progressValue} className="h-2" />
          </div>

          {/* Step 1: Language Selection */}
          {step === 1 && (
            <Card>
              <CardHeader>
                <CardTitle className="text-2xl">الاستعلام عن التقرير الائتماني</CardTitle>
                <CardDescription>تسمح لك هذه الخدمة بالاستعلام عن التقرير الائتماني الخاص بك</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <Label className="text-lg mb-4 block">اختر اللغة:</Label>
                  <RadioGroup value={language} onValueChange={setLanguage}>
                    <div className="flex items-center space-x-2 space-x-reverse mb-3">
                      <RadioGroupItem value="ar" id="ar" />
                      <Label htmlFor="ar" className="cursor-pointer">العربية</Label>
                    </div>
                    <div className="flex items-center space-x-2 space-x-reverse">
                      <RadioGroupItem value="en" id="en" />
                      <Label htmlFor="en" className="cursor-pointer">الإنجليزية</Label>
                    </div>
                  </RadioGroup>
                </div>
                <Button onClick={handleStep1Submit} className="w-full" size="lg">
                  التالي
                </Button>
              </CardContent>
            </Card>
          )}

          {/* Step 2: Personal Information */}
          {step === 2 && (
            <Card>
              <CardHeader>
                <CardTitle className="text-2xl">بيانات الطلب</CardTitle>
                <CardDescription>قم باستكمال بيانات الطلب</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <Label htmlFor="idCard">صورة البطاقة الأمامية (واضحة)</Label>
                  <div className="mt-2">
                    <input
                      type="file"
                      ref={fileInputRef}
                      onChange={handleFileChange}
                      accept="image/*"
                      className="hidden"
                      id="idCard"
                    />
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() => fileInputRef.current?.click()}
                      className="w-full"
                    >
                      <Upload className="ml-2 h-4 w-4" />
                      {idCardFile ? "تم رفع الصورة" : "رفع صورة البطاقة"}
                    </Button>
                    {idCardPreview && (
                      <img src={idCardPreview} alt="Preview" className="mt-4 max-h-48 mx-auto rounded border" />
                    )}
                  </div>
                </div>

                <div>
                  <Label htmlFor="motherName">اسم الأم الأول</Label>
                  <Input
                    id="motherName"
                    value={formData.motherName}
                    onChange={(e) => setFormData({ ...formData, motherName: e.target.value })}
                    placeholder="أدخل اسم الأم الأول"
                  />
                </div>

                <div>
                  <Label htmlFor="idNumber">الرقم القومي</Label>
                  <Input
                    id="idNumber"
                    value={formData.idNumber}
                    onChange={(e) => setFormData({ ...formData, idNumber: e.target.value })}
                    placeholder="أدخل الرقم القومي"
                  />
                </div>

                <div>
                  <Label htmlFor="email">البريد الإلكتروني</Label>
                  <Input
                    id="email"
                    type="email"
                    value={formData.email}
                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                    placeholder="أدخل البريد الإلكتروني"
                  />
                </div>

                <div>
                  <Label htmlFor="phoneNumber">رقم الهاتف <span className="text-red-500">*</span></Label>
                  <Input
                    id="phoneNumber"
                    value={formData.phoneNumber}
                    onChange={(e) => setFormData({ ...formData, phoneNumber: e.target.value })}
                    placeholder="أدخل رقم الهاتف"
                  />
                </div>

                <div className="border-t pt-4">
                  <Label htmlFor="couponCode">كود الخصم (اختياري)</Label>
                  <div className="flex gap-2 mt-2">
                    <Input
                      id="couponCode"
                      value={couponCode}
                      onChange={(e) => setCouponCode(e.target.value.toUpperCase())}
                      placeholder="أدخل كود الخصم"
                      className="flex-1"
                    />
                    <Button
                      type="button"
                      onClick={handleApplyCoupon}
                      variant="outline"
                      disabled={validateCouponQuery.isFetching}
                    >
                      {validateCouponQuery.isFetching ? (
                        <Loader2 className="h-4 w-4 animate-spin" />
                      ) : (
                        "تطبيق"
                      )}
                    </Button>
                  </div>
                  {couponError && (
                    <p className="text-sm text-red-500 mt-1">{couponError}</p>
                  )}
                  {discount > 0 && (
                    <div className="mt-3 p-3 bg-green-50 dark:bg-green-900/20 rounded-lg">
                      <p className="text-sm text-green-700 dark:text-green-400">
                        ✓ تم تطبيق خصم {discount} جنيه
                      </p>
                      <div className="flex justify-between mt-2 text-sm">
                        <span>السعر الأصلي:</span>
                        <span className="line-through">180 جنيه</span>
                      </div>
                      <div className="flex justify-between font-bold text-green-700 dark:text-green-400">
                        <span>السعر بعد الخصم:</span>
                        <span>{finalPrice} جنيه</span>
                      </div>
                    </div>
                  )}
                </div>

                <div className="flex gap-4">
                  <Button onClick={() => setStep(1)} variant="outline" className="flex-1">
                    <ArrowRight className="ml-2 h-4 w-4" />
                    رجوع
                  </Button>
                  <Button
                    onClick={handleStep2Submit}
                    className="flex-1"
                    disabled={createRequestMutation.isPending}
                  >
                    {createRequestMutation.isPending ? (
                      <>
                        <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                        جاري الإرسال...
                      </>
                    ) : (
                      "التالي"
                    )}
                  </Button>
                </div>
              </CardContent>
            </Card>
          )}

          {/* Step 3: Payment */}
          {step === 3 && requestId && (
            <Card>
              <CardHeader>
                <CardTitle className="text-2xl">الدفع</CardTitle>
                <CardDescription>اختر طريقة الدفع المفضلة</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center mb-6">
                  <p className="text-lg mb-2">رمز التحقق الخاص بك:</p>
                  <p className="text-3xl font-bold text-cyan-600">{verificationCode}</p>
                  <p className="text-sm text-gray-600 mt-2">احتفظ بهذا الرمز لتتبع طلبك</p>
                </div>
                <Button
                  onClick={() => setLocation(`/payment/${requestId}`)}
                  className="w-full"
                  size="lg"
                >
                  المتابعة للدفع
                </Button>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
}
