import { useState } from "react";
import { Link, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { trpc } from "@/lib/trpc";
import { toast } from "sonner";
import { APP_LOGO } from "@/const";
import { Search, Loader2 } from "lucide-react";

export default function Track() {
  const [, setLocation] = useLocation();
  const [trackingCode, setTrackingCode] = useState("");
  const [isSearching, setIsSearching] = useState(false);

  const handleTrack = async () => {
    if (!trackingCode) {
      toast.error("الرجاء إدخال رمز التتبع");
      return;
    }

    setIsSearching(true);

    try {
      // Try to find by verification code
      const response = await fetch(`/api/trpc/request.getByVerificationCode?input=${encodeURIComponent(JSON.stringify({ code: trackingCode }))}`);
      
      if (response.ok) {
        setLocation(`/payment-result/${trackingCode}`);
      } else {
        toast.error("الطلب غير موجود");
      }
    } catch (error) {
      toast.error("حدث خطأ أثناء البحث");
      console.error(error);
    } finally {
      setIsSearching(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link href="/">
            <div className="flex items-center gap-3 cursor-pointer">
              <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
            </div>
          </Link>
          <Link href="/">
            <Button variant="outline">الرئيسية</Button>
          </Link>
        </div>
      </header>

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-2xl mx-auto">
          <Card>
            <CardHeader>
              <CardTitle className="text-2xl text-center">تتبع الطلب</CardTitle>
              <CardDescription className="text-center">
                أدخل رمز التحقق الخاص بك لمتابعة حالة طلبك
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <Label htmlFor="trackingCode">رمز التحقق</Label>
                <Input
                  id="trackingCode"
                  value={trackingCode}
                  onChange={(e) => setTrackingCode(e.target.value)}
                  placeholder="أدخل رمز التحقق المكون من 6 أرقام"
                  className="text-center text-2xl tracking-widest"
                  maxLength={6}
                  onKeyPress={(e) => {
                    if (e.key === "Enter") {
                      handleTrack();
                    }
                  }}
                />
              </div>

              <Button
                onClick={handleTrack}
                className="w-full"
                size="lg"
                disabled={isSearching}
              >
                {isSearching ? (
                  <>
                    <Loader2 className="ml-2 h-5 w-5 animate-spin" />
                    جاري البحث...
                  </>
                ) : (
                  <>
                    <Search className="ml-2 h-5 w-5" />
                    تتبع الطلب
                  </>
                )}
              </Button>

              <div className="border-t pt-6">
                <h4 className="font-bold mb-4 text-center">لا تملك رمز تحقق؟</h4>
                <p className="text-sm text-gray-600 text-center mb-4">
                  يمكنك إنشاء طلب جديد للحصول على تقريرك الائتماني
                </p>
                <Link href="/request">
                  <Button variant="outline" className="w-full">
                    إنشاء طلب جديد
                  </Button>
                </Link>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
