import { eq, desc } from "drizzle-orm";
import { drizzle } from "drizzle-orm/mysql2";
import { InsertUser, users, statistics, requests, paymentMethods, messages, reviews, coupons, InsertRequest, InsertPaymentMethod, InsertMessage, InsertReview, Coupon, InsertCoupon } from "../drizzle/schema";
import { ENV } from './_core/env';

let _db: ReturnType<typeof drizzle> | null = null;

export async function getDb() {
  if (!_db && process.env.DATABASE_URL) {
    try {
      _db = drizzle(process.env.DATABASE_URL);
    } catch (error) {
      console.warn("[Database] Failed to connect:", error);
      _db = null;
    }
  }
  return _db;
}

export async function upsertUser(user: InsertUser): Promise<void> {
  if (!user.openId) {
    throw new Error("User openId is required for upsert");
  }

  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot upsert user: database not available");
    return;
  }

  try {
    const values: InsertUser = {
      openId: user.openId,
    };
    const updateSet: Record<string, unknown> = {};

    const textFields = ["name", "email", "loginMethod"] as const;
    type TextField = (typeof textFields)[number];

    const assignNullable = (field: TextField) => {
      const value = user[field];
      if (value === undefined) return;
      const normalized = value ?? null;
      values[field] = normalized;
      updateSet[field] = normalized;
    };

    textFields.forEach(assignNullable);

    if (user.lastSignedIn !== undefined) {
      values.lastSignedIn = user.lastSignedIn;
      updateSet.lastSignedIn = user.lastSignedIn;
    }
    if (user.role !== undefined) {
      values.role = user.role;
      updateSet.role = user.role;
    } else if (user.openId === ENV.ownerOpenId) {
      values.role = 'admin';
      updateSet.role = 'admin';
    }

    if (!values.lastSignedIn) {
      values.lastSignedIn = new Date();
    }

    if (Object.keys(updateSet).length === 0) {
      updateSet.lastSignedIn = new Date();
    }

    await db.insert(users).values(values).onDuplicateKeyUpdate({
      set: updateSet,
    });
  } catch (error) {
    console.error("[Database] Failed to upsert user:", error);
    throw error;
  }
}

export async function getUserByOpenId(openId: string) {
  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot get user: database not available");
    return undefined;
  }

  const result = await db.select().from(users).where(eq(users.openId, openId)).limit(1);

  return result.length > 0 ? result[0] : undefined;
}

// Request helpers
export async function createRequest(data: InsertRequest): Promise<number> {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  const result = await db.insert(requests).values(data);
  // Get the last inserted ID
  const insertedId = (result as any)[0]?.insertId || (result as any).insertId;
  return Number(insertedId);
}

export async function getRequestById(id: number) {
  const db = await getDb();
  if (!db) return undefined;
  
  const result = await db.select().from(requests).where(eq(requests.id, id)).limit(1);
  return result.length > 0 ? result[0] : undefined;
}

export async function getRequestByIdNumber(idNumber: string) {
  const db = await getDb();
  if (!db) return undefined;
  
  const result = await db.select().from(requests).where(eq(requests.idNumber, idNumber)).orderBy(desc(requests.createdAt)).limit(1);
  return result.length > 0 ? result[0] : undefined;
}

export async function getRequestByVerificationCode(code: string) {
  const db = await getDb();
  if (!db) return undefined;
  
  const result = await db.select().from(requests).where(eq(requests.verificationCode, code)).limit(1);
  return result.length > 0 ? result[0] : undefined;
}

export async function getAllRequests() {
  const db = await getDb();
  if (!db) return [];
  
  return await db.select().from(requests).orderBy(desc(requests.createdAt));
}

export async function updateRequest(id: number, data: Partial<InsertRequest>) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.update(requests).set(data).where(eq(requests.id, id));
}

// Payment method helpers
export async function getActivePaymentMethods() {
  const db = await getDb();
  if (!db) return [];
  
  return await db.select().from(paymentMethods).where(eq(paymentMethods.isActive, true));
}

export async function getAllPaymentMethods() {
  const db = await getDb();
  if (!db) return [];
  
  return await db.select().from(paymentMethods);
}

export async function createPaymentMethod(data: InsertPaymentMethod) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.insert(paymentMethods).values(data);
}

export async function updatePaymentMethod(id: number, data: Partial<InsertPaymentMethod>) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.update(paymentMethods).set(data).where(eq(paymentMethods.id, id));
}

export async function deletePaymentMethod(id: number) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.delete(paymentMethods).where(eq(paymentMethods.id, id));
}

// Message helpers
export async function sendMessage(data: InsertMessage) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.insert(messages).values(data);
}

export async function createMessage(data: InsertMessage) {
  return sendMessage(data);
}

// Statistics helpers
export async function getStatistics() {
  const db = await getDb();
  if (!db) return null;
  
  const result = await db.select().from(statistics).limit(1);
  return result.length > 0 ? result[0] : null;
}

export async function updateStatistics(data: { totalRequests?: number; totalRevenue?: number; totalUsers?: number }) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  const current = await getStatistics();
  if (!current) {
    // Create initial stats
    await db.insert(statistics).values({
      totalRequests: data.totalRequests || 0,
      totalRevenue: data.totalRevenue || 0,
      totalUsers: data.totalUsers || 0,
    });
  } else {
    // Update existing stats
    await db.update(statistics)
      .set(data)
      .where(eq(statistics.id, current.id));
  }
}

// Review helpers
export async function createReview(review: InsertReview) {
  const db = await getDb();
  if (!db) return;
  await db.insert(reviews).values(review);
}

export async function getAllReviews() {
  const db = await getDb();
  if (!db) return [];
  const result = await db.select().from(reviews).orderBy(desc(reviews.createdAt));
  return result;
}

// ===== Coupon Functions =====

export async function getCouponByCode(code: string): Promise<Coupon | undefined> {
  const db = await getDb();
  if (!db) return undefined;
  const result = await db.select().from(coupons).where(eq(coupons.code, code)).limit(1);
  return result.length > 0 ? result[0] : undefined;
}

export async function validateCoupon(code: string): Promise<{ valid: boolean; coupon?: Coupon; message?: string }> {
  const coupon = await getCouponByCode(code);
  
  if (!coupon) {
    return { valid: false, message: "كود الخصم غير صحيح" };
  }
  
  if (coupon.isActive === 0) {
    return { valid: false, message: "كود الخصم غير نشط" };
  }
  
  if (coupon.expiresAt && new Date(coupon.expiresAt) < new Date()) {
    return { valid: false, message: "كود الخصم منتهي الصلاحية" };
  }
  
  if (coupon.usageLimit && coupon.usedCount >= coupon.usageLimit) {
    return { valid: false, message: "تم استخدام كود الخصم بالكامل" };
  }
  
  return { valid: true, coupon };
}

export async function incrementCouponUsage(code: string) {
  const db = await getDb();
  if (!db) return;
  const coupon = await getCouponByCode(code);
  if (coupon) {
    await db.update(coupons)
      .set({ usedCount: coupon.usedCount + 1 })
      .where(eq(coupons.code, code));
  }
}

export async function getAllCoupons() {
  const db = await getDb();
  if (!db) return [];
  const result = await db.select().from(coupons).orderBy(desc(coupons.createdAt));
  return result;
}

export async function createCoupon(coupon: InsertCoupon) {
  const db = await getDb();
  if (!db) return;
  await db.insert(coupons).values(coupon);
}

export async function updateCoupon(id: number, updates: Partial<Coupon>) {
  const db = await getDb();
  if (!db) return;
  await db.update(coupons).set(updates).where(eq(coupons.id, id));
}

export async function deleteCoupon(id: number) {
  const db = await getDb();
  if (!db) return;
  await db.delete(coupons).where(eq(coupons.id, id));
}

export async function getMessagesByRequestId(requestId: number) {
  const db = await getDb();
  if (!db) return [];
  
  return await db.select().from(messages).where(eq(messages.requestId, requestId)).orderBy(desc(messages.createdAt));
}

export async function getUnreadAdminMessages() {
  const db = await getDb();
  if (!db) return [];
  
  return await db.select().from(messages).where(eq(messages.isRead, false));
}

export async function markMessageAsRead(id: number) {
  const db = await getDb();
  if (!db) throw new Error("Database not available");
  
  await db.update(messages).set({ isRead: true }).where(eq(messages.id, id));
}
