import { describe, expect, it } from "vitest";
import { appRouter } from "./routers";
import type { TrpcContext } from "./_core/context";

type AuthenticatedUser = NonNullable<TrpcContext["user"]>;

function createAdminContext(): TrpcContext {
  const user: AuthenticatedUser = {
    id: 1,
    openId: "admin-user",
    email: "admin@iscore.com",
    name: "Admin User",
    loginMethod: "manus",
    role: "admin",
    createdAt: new Date(),
    updatedAt: new Date(),
    lastSignedIn: new Date(),
  };

  return {
    user,
    req: {
      protocol: "https",
      headers: {},
    } as TrpcContext["req"],
    res: {
      clearCookie: () => {},
    } as TrpcContext["res"],
  };
}

function createPublicContext(): TrpcContext {
  return {
    user: null,
    req: {
      protocol: "https",
      headers: {},
    } as TrpcContext["req"],
    res: {
      clearCookie: () => {},
    } as TrpcContext["res"],
  };
}

describe("Request Management", () => {
  it("should create a new request with verification code", async () => {
    const ctx = createPublicContext();
    const caller = appRouter.createCaller(ctx);

    const result = await caller.request.create({
      language: "ar",
      motherName: "فاطمة",
      idNumber: "12345678901234",
      email: "test@example.com",
      phoneNumber: "01234567890",
    });

    expect(result.success).toBe(true);
    expect(result.verificationCode).toBeDefined();
    expect(result.verificationCode?.length).toBe(6);
  });

  it("should allow admin to get all requests", async () => {
    const ctx = createAdminContext();
    const caller = appRouter.createCaller(ctx);

    const requests = await caller.request.getAll();

    expect(Array.isArray(requests)).toBe(true);
  });

  it("should allow admin to update request status", async () => {
    const ctx = createAdminContext();
    const caller = appRouter.createCaller(ctx);

    // First create a request
    const publicCaller = appRouter.createCaller(createPublicContext());
    const createResult = await publicCaller.request.create({
      language: "ar",
      motherName: "فاطمة",
      idNumber: "98765432109876",
      email: "test2@example.com",
    });

    // Get the request by verification code
    const request = await publicCaller.request.getByVerificationCode({
      code: createResult.verificationCode || "",
    });

    // Update status
    const updateResult = await caller.request.updateStatus({
      requestId: request.id,
      status: "paid",
    });

    expect(updateResult.success).toBe(true);
  });
});

describe("Payment Methods", () => {
  it("should get active payment methods", async () => {
    const ctx = createPublicContext();
    const caller = appRouter.createCaller(ctx);

    const methods = await caller.paymentMethod.getActive();

    expect(Array.isArray(methods)).toBe(true);
    expect(methods.length).toBeGreaterThan(0);
  });

  it("should allow admin to create payment method", async () => {
    const ctx = createAdminContext();
    const caller = appRouter.createCaller(ctx);

    const result = await caller.paymentMethod.create({
      nameAr: "طريقة اختبار",
      nameEn: "Test Method",
      type: "test",
      details: "تفاصيل الاختبار",
    });

    expect(result.success).toBe(true);
  });
});

describe("Messages", () => {
  it("should send message from user", async () => {
    const ctx = createPublicContext();
    const caller = appRouter.createCaller(ctx);

    // First create a request
    const createResult = await caller.request.create({
      language: "ar",
      motherName: "فاطمة",
      idNumber: "11111111111111",
      email: "test3@example.com",
    });

    const request = await caller.request.getByVerificationCode({
      code: createResult.verificationCode || "",
    });

    // Send message
    const messageResult = await caller.message.send({
      requestId: request.id,
      message: "مرحباً، أحتاج مساعدة",
    });

    expect(messageResult.success).toBe(true);
  });

  it("should get messages by request", async () => {
    const ctx = createPublicContext();
    const caller = appRouter.createCaller(ctx);

    // Create request and send message
    const createResult = await caller.request.create({
      language: "ar",
      motherName: "فاطمة",
      idNumber: "22222222222222",
      email: "test4@example.com",
    });

    const request = await caller.request.getByVerificationCode({
      code: createResult.verificationCode || "",
    });

    await caller.message.send({
      requestId: request.id,
      message: "رسالة اختبار",
    });

    // Get messages
    const messages = await caller.message.getByRequest({
      requestId: request.id,
    });

    expect(Array.isArray(messages)).toBe(true);
    expect(messages.length).toBeGreaterThan(0);
  });
});
